<?php 

class config {
	
	private static array $fields=array(
			'name'=>validator::REQUIRED,
			'description'=>validator::REQUIRED,
			'type'=>validator::REQUIRED,
			'enumvalues'=>validator::ANY,
			'minimum'=>validator::FLOAT,
			'maximum'=>validator::FLOAT,
			'defaultvalue'=>validator::ANY,
			'value'=>validator::ANY,
	);
	
	private static array $helpTexts=array(
			'name'=>'The field name',
			'description'=>'Short description of the field',
			'type'=>'string, enum, etc.',
			'enumvalues'=>'A list of values',
			'minimum'=>'For numeric values, the minimum',
			'maximum'=>'For numeric values, the maximum',
			'defaultvalue'=>'The default value of the field',
			'value'=>'The current value',
	);
	
	public static function getFieldValidations(): array {
		return self::$fields;
	}
	public static function getFieldHelpTexts(): array {
		return self::$helpTexts;
	}
	
	public static function canCreate(): bool {
		return session::isAdmin();
	}

    /**
     * @param $id
     * @return bool
     * @noinspection PhpUnusedParameterInspection
     */
    public static function canUpdate($id): bool {
		return session::isAdmin();
	}

	/**
	 * @throws ServerException
	 * @throws BadRequestException
	 */
	static function get($name): ?string {
		$result=database::queryGetOne('SELECT value FROM config WHERE name=:name', array(':name'=>$name));
		if(!$result){ return null; }
		return trim($result['value']);
	}

	/**
	 * @param $name
	 * @param $value
	 * @return bool
	 * @throws BadRequestException
	 * @throws ForbiddenException
	 * @throws ServerException
	 */
	public static function set($name, $value): bool {
		if(!session::isAdmin()){ throw new ForbiddenException('Only administrators can see or change the configuration'); }
		$item=database::queryGetOne('SELECT * FROM config WHERE name=:name', array(':name'=>$name));
		if(!$item){ throw new BadRequestException('No config item called '.$name); }
		//TODO Validate!
		database::query('UPDATE config SET value=:value WHERE name=:name', array(':name'=>$name, ':value'=>trim($value)));
		if('core_backgroundimage_allow'==$name){
			backgroundimage::deleteInvalidUserConfigsAfterConfigChange();
		}
		return true;
	}

    /**
     * @param array $request
     * @return array|null
     * @throws ForbiddenException
	 * @throws ServerException
	 * @throws BadRequestException
     * @noinspection PhpUnusedParameterInspection
     */
	public static function getAll(array $request=array()): ?array{
		if(!session::isAdmin()){ throw new ForbiddenException('Only administrators can see the configuration'); }
		$result=database::queryGetAll('SELECT * FROM config ORDER BY name ');
		if(!$result){ return null; }
		return $result;
	}

    /**
     * @param $id
     * @throws BadRequestException
     * @noinspection PhpUnusedParameterInspection
     */
	static function getById($id){
		throw new BadRequestException('getById not implemented on config');
	}

    /**
     * @param $name
     * @throws BadRequestException
     * @noinspection PhpUnusedParameterInspection
     */
	static function getByName($name){
		throw new BadRequestException('getByName not implemented on config');
	}

	/**
	 * @param $key
	 * @param $value
	 * @param array $request
	 * @throws BadRequestException
	 * @noinspection PhpUnusedParameterInspection
	 */
	static function getByProperty($key,$value,$request=array()){
		throw new BadRequestException('getByProperty not implemented on config');
	}

	/**
	 * @param $key
	 * @param $value
	 * @param array $request
	 * @throws BadRequestException
	 * @noinspection PhpUnusedParameterInspection
	 */
	static function getByProperties($keyValuePairs,$request=array()){
		throw new BadRequestException('getByProperties not implemented on config');
	}

	/**
     * Returns the IceBear document root (the full path of the directory above api and client), without trailing slash.
     * @return string
     */
    public static function getWwwRoot(): string {
        return rtrim(dirname(__DIR__, 2),'/');
    }

    /**
     * Whether this looks like an IceBox managed IceBear installation.
     * @return bool
     */
    public static function isIceBox(): bool {
        $isIceBox=null;
        $sessionExists=session::exists();
        if($sessionExists){
            $isIceBox=session::get('isIceBox');
        }
		if(is_null($isIceBox) && isset($_SERVER['HTTP_HOST'])){
			$iceBoxHostname='.icebox.icebear.fi';
			if(str_ends_with(rtrim($_SERVER['HTTP_HOST'], '/'), $iceBoxHostname)){
				$isIceBox=true;
				session::set('isIceBox', true);
			}
		}
        if(is_null($isIceBox)) {
            $isIceBox = file_exists(config::getWwwRoot().'/isIceBox');
            if($sessionExists){
                session::set('isIceBox', $isIceBox);
            }
        }
        return $isIceBox;
    }

	/**
	 * Returns the unique subdomain part ZZZZ for a given IceBox installation, if installed at /var/www/icebear-ZZZZZ,
	 * otherwise returns false. This is used in the storage mount, the www-root, the database name, etc.
	 * @return string|null
	 */
    public static function getIceBoxName(): ?string {
        $iceBoxName=null;
        $sessionExists=session::exists();
        if($sessionExists){
            $iceBoxName=session::get('iceBoxName');
        }
		if(is_null($iceBoxName) && isset($_SERVER['HTTP_HOST'])){
			$host=$_SERVER['HTTP_HOST'];
			$domain='.icebox.icebear.fi';
			if(str_ends_with($host,$domain)){
				$name=str_replace($domain,'',$host);
				if(preg_match('/^[a-z0-9_-]+$/', $name)){
					$iceBoxName=$name;
				}
			}

		}
        if(is_null($iceBoxName)){
            if(!static::isIceBox()){
                $iceBoxName=null;
            } else {
                $path=rtrim(dirname(__DIR__,2),'/');
                if(!str_starts_with($path, '/var/www/icebear-')){
                    //Shouldn't happen.
                    $iceBoxName=null;
                } else {
                    $iceBoxName=str_replace('/var/www/icebear-','',$path);
                }
            }
        }
        if(session::exists()){
            session::set('iceBoxName', $iceBoxName);
        }
        return $iceBoxName;
    }

	/**
	 * @throws BadRequestException
	 * @throws ServerException
	 */
	public static function getDatabaseVersion(): string {
		$databaseVersion=config::get('core_icebearversion');
		if(empty($databaseVersion)){
			if(Log::isInited()){
				Log::write(Log::LOGLEVEL_WARN,'Database version not set. Assuming and setting 1.0');
			}
			database::query(
				"INSERT INTO config(name,description,type,defaultvalue,value) 
                    VALUES('core_icebearversion','IceBear database version number','text','','1.0')"
			);
			$databaseVersion='1.0';
		}
		return trim($databaseVersion);
	}

	public static function getCodeVersion(): string {
		$fileName=__DIR__.'/../../conf/codeversion';
		if(!file_exists($fileName)){
			throw new ServerException('IceBear directory does not contain file conf/codeversion.');
		}
		$contents=@file_get_contents($fileName);
		return trim($contents);
	}

}