<?php class RigakuImport extends Device {

    const USERNAME='rigakuimporter';
    const USERFULLNAME='Rigaku Importer';

    const SCORING_SYSTEM_NAME='Rigaku';

    const BARCODE_PATTERN='LLNNNNNN'; // Two letters, six numbers, e.g., UR000123

    const RIGAKU_SCORES= array(
        array('color'=>'ffffff', 'index'=>'0', 'hotkey'=>'0', 'name'=>'Clear'),
        array('color'=>'aeaeae', 'index'=>'1', 'hotkey'=>'9', 'name'=>'Matter'),
        array('color'=>'c800ff', 'index'=>'2', 'hotkey'=>'8', 'name'=>'Phase separation'),
        array('color'=>'ff0041', 'index'=>'3', 'hotkey'=>'7', 'name'=>'Heavy precipitation'),
        array('color'=>'ff8200', 'index'=>'4', 'hotkey'=>'6', 'name'=>'Precipitation'),
        array('color'=>'009b00', 'index'=>'5', 'hotkey'=>'5', 'name'=>'Spherulites'),
        array('color'=>'00ff1b', 'index'=>'6', 'hotkey'=>'4', 'name'=>'Clusters'),
        array('color'=>'00ffeb', 'index'=>'7', 'hotkey'=>'3', 'name'=>'Crystalline'),
        array('color'=>'94c6ff', 'index'=>'8', 'hotkey'=>'2', 'name'=>'Micro-crystals'),
        array('color'=>'0064fa', 'index'=>'9', 'hotkey'=>'1', 'name'=>'Crystals')
    );

    /**
     * @param $params
     * @return array
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function createScores($params){
        Log::debug('In RigakuImport::createScores');
            foreach($params as $k=>$v){
                if(is_array($v)){
                    Log::debug('- '.$k.'=Array');
                } else {
                    Log::debug('- '.$k.'='.$v);
                }
            }
        if(!isset($params['name'])){
            Log::info('Setting default scoring system name '.static::SCORING_SYSTEM_NAME);
            $params['name']=static::SCORING_SYSTEM_NAME;
        }
        if(!isset($params['scores'])){
            Log::info('Setting default Rigaku scores');
            $params['scores']=static::RIGAKU_SCORES;
        }
        $scoringSystem=ImageScoring::createScoringSystem($params);
        Log::debug('Returning from RigakuImport::createScores');
        return $scoringSystem;
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws BadRequestException
     */
    public static function getScores(){
        Log::debug('In RigakuImport::getScores');
        $scores=ImageScoring::getScoresWithScoringSystemName(static::SCORING_SYSTEM_NAME);
        Log::debug('Returning from RigakuImport::getScores');
        return $scores;
    }

    /**
     * @param $params
     * @return mixed
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function getImagingProfiles($params){
        Log::debug('In getImagingProfiles');
        $profiles=static::createImagingProfiles($params);
        Log::debug('Returning from getImagingProfiles');
        return $profiles;
    }

    /**
     * @param $params
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     * @throws ForbiddenException
     */
    public static function createImagingProfiles($params){
        Log::debug('In createImagingProfiles');
        static::require('profiles',$params);
        session::becomeAdmin();
        $sharedProjectId=baseproject::getSharedProjectId();
        foreach($params['profiles'] as $key=>&$rigakuProfile){
            static::require(['label','name','light'], $rigakuProfile);
            $label=$rigakuProfile['label'];
            $profileName=$rigakuProfile['name'];
            $profileVersionName=$profileName.'_v1';
            if($rigakuProfile['label']!==$key){
                throw new BadRequestException("Imaging profiles: Key $key does not match profile label/shortcode $label");
            }
            $iceBearProfile=imagingparameters::getByProperties([
                'name'=>$profileName,
                'manufacturer'=>'Rigaku'
            ]);

            if(!$iceBearProfile){
                $iceBearProfile=imagingparameters::create([
                    'name'=>$profileName,
                    'manufacturer'=>'Rigaku',
                    'manufacturerdatabaseid'=>0,
                    'projectid'=>$sharedProjectId
                ])['created'];
            } else {
                $iceBearProfile=$iceBearProfile['rows'][0];
            }

            $iceBearProfileVersion=imagingparametersversion::getByProperties([
                'name'=>$profileVersionName,
                'imagingparametersid'=>$iceBearProfile['id']
            ]);

            if(!$iceBearProfileVersion){
                $iceBearProfileVersion=imagingparametersversion::create([
                    'name'=>$profileVersionName,
                    'manufacturerdatabaseid'=>0,
                    'imagingparametersid'=>$iceBearProfile['id'],
                    'projectid'=>$sharedProjectId
                ])['created'];
            } else {
                $iceBearProfileVersion=$iceBearProfileVersion['rows'][0];
            }
            imagingparameters::update($iceBearProfile['id'], ['currentversionid'=>$iceBearProfileVersion['id']]);

            $rigakuProfile['profile']=$iceBearProfile;
            $rigakuProfile['profileVersion']=$iceBearProfileVersion;
        }
        Log::debug('Returning from createImagingProfiles');
        return $params;
    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function createUser($user){
        Log::debug('In RigakuImport::createUser');
        foreach($user as $k=>$v){
            Log::debug('- '.$k.'='.$v);
        }
        static::require('username',$user);
        $username=$user['username'];
        $usernameSanitized=preg_replace('/[^A-Za-z0-9_.-]/','',$username);
        if(!$username!==$usernameSanitized){
            Log::warning('Username contains illegal characters. Sanitized.');
            Log::warning("Supplied: $username");
            Log::warning("Sanitized: $usernameSanitized");
            $user['username']=$usernameSanitized;
            $username=$usernameSanitized;
        }
        if(!isset($user['fullName'])){
            Log::info('User fullName not set, setting to username: '.$username);
            $user['fullName']=$username;
        }
        $user=UserManagement::createUser($user);
        Log::debug('Returning from RigakuImport::createUser');
        return $user;
    }

    /**
     * @param $user
     * @return array|mixed
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function getUser($user){
        return static::createUser($user);
    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function createImager($params){
        Log::debug('In RigakuImport::createImager');
        static::require(['name','nominalTemperature'], $params);
        $params['manufacturerName']='Rigaku';
        $imager=ImagingManagement::createImager($params);
        Log::debug('Returning from FormulatrixImport::createImager');
        return $imager;
    }

    /**
     * @param $plateType
     * @return array|mixed
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function getPlateType($plateType){
        return static::createPlateType($plateType);
    }

    /**
     * @param $plateType
     * @return array|mixed
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function createPlateType($plateType){
        return PlateImport::createPlateType($plateType);
    }

    /**
     * @param $params
     * @return array
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function getScreenByName($params){
        return PlateImport::getScreenByName($params);
    }

    /**
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws ServerException
     * @throws NotFoundException
     */
    public static function promoteScreenToStandard($params){
        static::require('screenId', $params);
        session::becomeAdmin();
        $screenId=$params['screenId'];
        $screen=screen::getById($screenId);
        if(!$screen){ throw new NotFoundException('No screen with ID '.$screenId); }
        screen::update($screenId, [ 'isstandard'=>1 ]);
        return screen::getById($screenId);
    }

    /**
     * @param $params
     * @return array|mixed
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function getPlateByBarcode($params){
        Log::debug('In getPlateByBarcode');
        static::require('barcode',$params);
        $barcode=$params['barcode'];
        Log::debug("Plate barcode is $barcode");
        session::becomeAdmin();
        $plate=plate::getByName($barcode);
        session::revokeAdmin();
        Log::debug('Returning from getPlateByBarcode');
        return $plate;
    }

    /**
     * @param $params
     * @return mixed|null
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function getInspectionsForPlate($params){
        static::require('barcode', $params);
        $barcode=$params['barcode'];
        session::becomeAdmin();
        $plate=plate::getByName($params['barcode']);
        if(!$plate){ throw new NotFoundException("No plate with barcode $barcode"); }
        $inspections=plate::getimagingsessions($plate['id']);
        if(!$inspections){ return null; }
        Log::debug('Populating numImages');
        foreach ($inspections['rows'] as &$inspection){
            $images=imagingsession::getdropimages($inspection['id']);
            if(!$images || !isset($images['total'])){
                $inspection['numImages']=0;
            } else {
                $inspection['numImages']=$images['total'];
            }
        }
        Log::debug('Returning from getInspectionsForPlate');
        return $inspections['rows'];
    }

    /**
     * @param $params
     * @return mixed
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function createInspection($params){
        Log::debug('In createInspection');
        static::require('plateBarcode',$params);
        static::require('imagedDateTime',$params);
        static::require('profileName',$params);
        static::require('lightType',$params);
        static::require('imagerName',$params);
        static::require('temperature',$params);
        foreach ($params as $key=>$value){
            $$key=$value;
        }
        session::becomeAdmin();

        /** @var string $lightType */
        /** @var string $temperature */
        /** @var string $imagerName */
        /** @var string $imagedDateTime */
        /** @var string $plateBarcode */
        /** @var string $profileName */

        if(!is_numeric($temperature)){
            throw new BadRequestException('Temperature must be numeric');
        }

        $imager=imager::getByName($imagerName);
        if(!$imager){
            throw new NotFoundException("No imager called $imagerName");
        }
        $imagerId=$imager['id'];

        if(!preg_match('/^\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d$/', $imagedDateTime)){
            throw new BadRequestException('imagedDateTime must be a 24-hour GMT datetime in YYYY-MM-DD hh:mm:ss format');
        }

        $plate=plate::getByName($plateBarcode);
        if(!$plate){
            throw new NotFoundException('No plate with barcode '.$plateBarcode);
        }
        $plateId=$plate['id'];

        $profile=imagingparameters::getByProperties([
            'name'=>$profileName,
            'manufacturer'=>'Rigaku'
        ]);
        if(!$profile || !isset($profile['rows'])){
            throw new NotFoundException("No imaging profile called $profileName");
        } else if(1*($profile['total'])!==1){
            throw new ServerException($profile['total']. " Rigaku imaging profiles found called $profileName. Should be only 1.");
        }
        $profile=$profile['rows'][0];
        $profileId=$profile['id'];
        $profileVersionId=$profile['currentversionid'];
        if(is_null($profileVersionId)){
            throw new ServerException("imagingparameters $profileId ($profileName) has null currentversionid");
        }

        $inspectionParameters=[
            'plateid'=>$plateId,
            'imageddatetime'=>$imagedDateTime,
            'imagingparametersversionid'=>$profileVersionId
        ];
        Log::debug('Looking for existing imagingsession with parameters:');
        foreach ($inspectionParameters as $k=>$v){
            Log::debug("$k: $v");
        }
        $inspectionName=$plateBarcode.'_'.str_replace(' ','_',$imagedDateTime).'_profile'.$profileId;
        $inspection=imagingsession::getByName($inspectionName);
        if($inspection){
            Log::debug('Imaging session exists, returning it');
        } else {
            Log::info('Imaging session does not exist, creating it');
            $inspectionParameters['name']=$inspectionName;
            $inspectionParameters['manufacturerdatabaseid']=0;
            $inspectionParameters['imagerid']=$imagerId;
            $inspectionParameters['lighttype']=$lightType;
            $inspectionParameters['temperature']=$temperature;
            $inspection=imagingsession::create($inspectionParameters)['created'];
        }

        Log::debug('Returning from createInspection');
        return $inspection;
    }

    /**
     * @param $plate
     * @return mixed
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws NotModifiedException
     * @throws ServerException
     */
    public static function createPlate($plate){
        $plate['scoringSystemName']=self::SCORING_SYSTEM_NAME;
        return PlateImport::createPlate($plate);
    }

    /**
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public static function createImages($params){
        Log::debug('In createImages');
        return ImagingManagement::createImages($params);
    }

    /**
     * @param $params
     * @return array|null
     * @throws BadRequestException
     * @throws ServerException
     */
    public static function getProject($params){
        static::require('name', $params);
        return PlateImport::getProjectByName($params);
    }

    /**
     * @param $params
     * @return mixed
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function createProject($params){
        return PlateImport::createProject($params);
    }

    /**
     * @param $params
     * @return array[]
     * @throws BadRequestException
     */
    public static function updateDiskSpaces($params){
        $updatedDrives=Device::updateDiskSpaces($params);
        $content=array(
            'total'=>count($updatedDrives),
            'rows'=>$updatedDrives
        );
        caching::setContent(diskusage::FILESYSTEM_CACHE_RIGAKU, $content, '+12 hours');
        return $content;
    }

    /**
     * @throws NotFoundException
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws NotModifiedException
     * @throws ServerException
     */
    public static function createScreen($params){
        if(!isset($params['projectId'])){
            session::becomeAdmin();
            $params['projectId']=baseproject::getSharedProjectId();
        }
        $screen=PlateImport::createScreen($params);
        if(empty($screen)){
            throw new Exception('createScreen returned empty');
        }
        return $screen;
    }

    /**
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public static function getPlateForReimport(){
        return PlateImport::getPlateForReimport(['imagerType'=>'Rigaku']);
    }

}