<?php

include_once 'DeviceTest.php';

class ImagingManagementTest extends DeviceTest {

    /**
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateImager(){
        $name='TestImager'.time();
        $imager=ImagingManagement::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100,
            'manufacturerName'=>'Test'
        ));
        self::assertEquals($name, $imager['name']);
        self::assertEquals(100, $imager['platecapacity']);
        self::assertEquals(20, $imager['temperature']);
        self::assertEquals('Test', $imager['manufacturer']);
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateImagerBlank(){
        self::expectException('BadRequestException');
        ImagingManagement::createImager([]);
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateImagerNoName(){
        self::expectException('BadRequestException');
        ImagingManagement::createImager(array(
            'nominalTemperature'=>20,
            'plateCapacity'=>100,
            'manufacturerName'=>'Test'
        ));
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateImagerNoTemperature(){
        self::expectException('BadRequestException');
        ImagingManagement::createImager(array(
            'name'=>'TestImager'.time(),
            'plateCapacity'=>100,
            'manufacturerName'=>'Test'
        ));
    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateImagerNonNumericTemperature(){
        self::expectExceptionMessage(ImagingManagement::IMAGER_BAD_TEMPERATURE);
        ImagingManagement::createImager(array(
            'name'=>'TestImager'.time(),
            'nominalTemperature'=>'Foo',
            'plateCapacity'=>100,
            'manufacturerName'=>'Test'
        ));
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateImagerNoCapacity(){
        self::expectException('BadRequestException');
        ImagingManagement::createImager(array(
            'name'=>'TestImager'.time(),
            'nominalTemperature'=>20,
            'manufacturerName'=>'Test'
        ));
    }

    /**
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateImagerNegativeCapacity(){
        self::expectExceptionMessage(ImagingManagement::IMAGER_BAD_CAPACITY);
        ImagingManagement::createImager(array(
            'name'=>'TestImager'.time(),
            'nominalTemperature'=>20,
            'plateCapacity'=>-5,
            'manufacturerName'=>'Test'
        ));
    }

    /**
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateImagerNonNumericCapacity(){
        self::expectExceptionMessage(ImagingManagement::IMAGER_BAD_CAPACITY);
        ImagingManagement::createImager(array(
            'name'=>'TestImager'.time(),
            'nominalTemperature'=>20,
            'plateCapacity'=>'Foo',
            'manufacturerName'=>'Test'
        ));
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateImagerNoManufacturer(){
        self::expectException('BadRequestException');
        ImagingManagement::createImager(array(
            'name'=>'TestImager'.time(),
            'nominalTemperature'=>20,
            'plateCapacity'=>100
        ));
    }

    /**
     * Test that a capacity mismatch between the supplied imager info and a pre-existing imager does not
     * change the latter's capacity or throw exceptions.
     *
     * Rationale: A Formulatrix imager might be detected as having 182 slots (RI182) but be down-licensed
     * to 90. It should be possible to change IceBear's record to 90 without breaking import or having the
     * importer change the capacity back.
     *
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateImagerExistsCapacityMismatch(){
        $name='TestImager'.time();
        ImagingManagement::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100,
            'manufacturerName'=>'Test'
        ));
        $imager=ImagingManagement::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>80,
            'manufacturerName'=>'Test'
        ));
        self::assertEquals(100, $imager['platecapacity']);
    }

    /**
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateImagerExistsTemperatureMismatch(){
        $name='TestImager'.time();
        ImagingManagement::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100,
            'manufacturerName'=>'Test'
        ));
        $imager=ImagingManagement::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>4,
            'plateCapacity'=>100,
            'manufacturerName'=>'Test'
        ));
        self::assertEquals(20, $imager['temperature']);
    }

}