<?php

include_once 'DeviceTest.php';

class RigakuImportTest extends DeviceTest {

    /**
     * @throws BadRequestException
     */
    public function testCreateScores(){
        $scoringSystem=RigakuImport::createScores([]);
        self::assertArrayHasKey('name',$scoringSystem);
        self::assertArrayHasKey('scores',$scoringSystem);
        self::assertEquals(RigakuImport::SCORING_SYSTEM_NAME, $scoringSystem['name']);
        self::assertSameSize(RigakuImport::RIGAKU_SCORES, $scoringSystem['scores']);
    }

    /**
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testGetScores(){
        RigakuImport::createScores([]);
        $scores=RigakuImport::getScores();
        self::assertNotEmpty($scores);
        self::assertCount(count(RigakuImport::RIGAKU_SCORES), $scores);
        self::assertArrayHasKey("id", $scores[0]);
        self::assertArrayHasKey("label", $scores[0]);
        self::assertArrayHasKey("color", $scores[0]);
        self::assertArrayHasKey("hotkey", $scores[0]);
        self::assertArrayHasKey("scoreindex", $scores[0]);
        $expectedLabels=array_column(RigakuImport::RIGAKU_SCORES,'name');
        $actualLabels=array_column($scores,'label');
        foreach ($expectedLabels as $expectedLabel){
            self::assertContains($expectedLabel, $actualLabels);
        }
    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateUser(){
        $username='user'.time();
        $user=['username'=>$username];
        $user=RigakuImport::createUser($user);
        self::assertEquals($username, $user['name']);
        self::assertEquals($username.'@bogus.bogus', $user['email']);
    }

    /**
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateUserWithSpaceInUsername(){
        $username='user '.time();
        $user=['username'=>$username];
        $user=RigakuImport::createUser($user);
        $username=str_replace(' ','',$username);
        self::assertEquals($username, $user['name']);
        self::assertEquals(strtolower($username).'@bogus.bogus', $user['email']);
    }

    /**
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateImager(){
        $name='imager'.time();
        $imager=RigakuImport::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100
        ));
        self::assertEquals($name, $imager['name']);
        self::assertEquals(20, $imager['temperature']);
        self::assertEquals(100, $imager['platecapacity']);
    }

    /**
     * A second call to createImager() should return the existing record
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateImagerExists(){
        $name='imager'.time();
        RigakuImport::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100
        ));
        $imager=RigakuImport::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100
        ));
        self::assertEquals($name, $imager['name']);
        self::assertEquals(20, $imager['temperature']);
        self::assertEquals(100, $imager['platecapacity']);
    }

    /**
     * Calling createImager() when the imager exists, with a different temperature, should throw an exception.
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateImagerExistsTemperatureMismatch(){
        $name='imager'.time();
        RigakuImport::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100
        ));
        $imager=RigakuImport::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>4,
            'plateCapacity'=>100
        ));
        self::assertEquals($name, $imager['name']);
        self::assertEquals(20, $imager['temperature']);
        self::assertEquals(100, $imager['platecapacity']);
    }

    /**
     * Calling createImager() with a different capacity should return the existing imager, and should not
     * change the capacity.
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateImagerExistsCapacityMismatch(){
        $name='imager'.time();
        RigakuImport::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>100
        ));
        $imager=RigakuImport::createImager(array(
            'name'=>$name,
            'nominalTemperature'=>20,
            'plateCapacity'=>200
        ));
        self::assertEquals($name, $imager['name']);
        self::assertEquals(20, $imager['temperature']);
        self::assertEquals(100, $imager['platecapacity']);
    }

    /**
     * @return void
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function testCreatePlateType(){
        $name='plateType'.time();
        RigakuImport::createPlateType([
            'name'=>$name,
            'rows'=>4,
            'cols'=>6,
            'subs'=>2
        ]);
        $plateType=platetype::getByName($name);
        self::assertNotEmpty($plateType);
        self::assertEquals(4, $plateType['rows']);
        self::assertEquals(6, $plateType['cols']);
        self::assertEquals(2, $plateType['subs']);
    }

    /**
     * @return void
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws NotFoundException
     * @throws ServerException
     */
    public static function testGetPlateType(){
        session::becomeAdmin();
        $name='plateType'.time();
        $sharedProject=project::getByName(baseproject::SHARED);
        $projectId=$sharedProject['id'];
        platetype::create([
            'name'=>$name,
            'projectid'=>$projectId,
            'rows'=>10,
            'cols'=>20,
            'subs'=>5,
            'dropmapping'=>'1,R'
        ]);
        session::revokeAdmin();
        $result=RigakuImport::getPlateType([
            'name'=>$name,
            'rows'=>10,
            'cols'=>20,
            'subs'=>5
        ]);
        self::assertEquals(10, $result['rows']);
        self::assertEquals(20, $result['cols']);
        self::assertEquals(5, $result['subs']);
    }

    public static function getScreenByName(){

    }

    /**
     * getScreenByName
     * promoteScreenToStandard
     * getPlateByBarcode
     * getInspectionsForPlate
     * createInspection
     * createPlate
     * createImages
     * createImage
     * getProject
     * createProject
     * getImagingProfiles
     * createImagingProfiles
     */

}