<?php

use PHPUnit\Framework\TestCase;

class configTest extends TestCase {

	protected ?string $host;

	/**
	 * @throws ServerException
	 * @throws NotFoundException
	 * @throws BadRequestException
	 */
	protected function setUp(): void {
		$this->host=$_SERVER['HTTP_HOST'];
		database::connect();
		database::begin();
		session::init(new DummySession());
	}

	protected function tearDown(): void {
		database::abort();
		session::destroy();
		@unlink(rtrim(config::getWwwRoot(),'/').'/isIceBox');
		$_SERVER['HTTP_HOST']=$this->host;
	}

	public function testGetFieldValidations(){
		$ret=config::getFieldValidations();
		$this->assertArrayHasKey('name',$ret);
		$this->assertEquals(validator::REQUIRED, $ret['name']);
	}

	public function testGetFieldHelpTexts(){
		$ret=config::getFieldHelpTexts();
		$this->assertArrayHasKey('name',$ret);
		$this->assertStringContainsString('name', $ret['name']);
	}

	/**
	 * @throws ServerException
	 * @throws BadRequestException
	 */
	public function testCanUpdate(): void {
		self::assertFalse(config::canUpdate(0));
		session::becomeAdmin();
		self::assertTrue(config::canUpdate(0));
	}

	/**
	 * @return void
	 * @throws BadRequestException
	 * @throws ServerException
	 */
	public function testCanCreate(): void {
		self::assertFalse(config::canCreate());
		session::becomeAdmin();
		self::assertTrue(config::canCreate());
	}

	public function testGet(){
		self::assertNull(config::get('bogus'));
		$iceBearVersion=file_get_contents(config::getWwwRoot().'/conf/codeversion');
		self::assertNotNull($iceBearVersion);
		self::assertEquals($iceBearVersion, config::get('core_icebearversion'));
	}

	/**
	 * @throws BadRequestException
	 * @throws ForbiddenException
	 * @throws ServerException
	 */
	public function testSetAsAdmin(){
		session::becomeAdmin();
		config::set('core_icebearversion','5.6.7');
		self::assertEquals('5.6.7', config::get('core_icebearversion'));
		self::expectException('BadRequestException');
		config::set('foo','bar');
	}
	/**
	 * @throws ServerException
	 * @throws BadRequestException
	 */
	public function testSetAsNonAdmin(){
		self::expectException('ForbiddenException');
		config::set('core_icebearversion','5.5.5');
	}

	/**
	 * @throws ForbiddenException
	 * @throws BadRequestException
	 * @throws ServerException
	 */
	public function testGetAll(){
		session::becomeAdmin();
		$config=config::getAll();
		self::assertIsArray($config);
		self::assertArrayHasKey('total',$config);
		self::assertArrayHasKey('rows',$config);
		$configNames=array_column($config['rows'],'name');
		self::assertContains('core_icebearversion', $configNames);
		session::revokeAdmin();
		self::expectException('ForbiddenException');
		config::getAll();
	}

	public function testGetWwwRoot(){
		$wwwRoot=realpath(__DIR__.'../../../..');
		$this->assertEquals($wwwRoot, config::getWwwRoot());
	}

	public function testGetByName(){
		$this->expectException('BadRequestException');
		$this->expectExceptionMessageMatches('/not implemented/');
		config::getByName('bogus');
	}

	public function testGetById(){
		$this->expectException('BadRequestException');
		$this->expectExceptionMessageMatches('/not implemented/');
		config::getById(123);
	}

	public function testGetByProperty(){
		$this->expectException('BadRequestException');
		$this->expectExceptionMessageMatches('/not implemented/');
		config::getByProperty('name','bogus');
	}

	public function testGetByProperties(){
		$this->expectException('BadRequestException');
		$this->expectExceptionMessageMatches('/not implemented/');
		config::getByProperties(['name'=>'bogus', 'foo'=>'bar']);
	}

	public function testIsIceBox(){
		self::assertFileDoesNotExist(rtrim(config::getWwwRoot(),'/').'/isIceBox');
		self::assertNull(session::get('isIceBox'));
		$isIceBox=config::isIceBox();
		self::assertFalse($isIceBox);
		self::assertFalse(session::get('isIceBox'));
		session::set('isIceBox',null);
		touch(rtrim(config::getWwwRoot(),'/').'/isIceBox');
		$isIceBox=config::isIceBox();
		self::assertTrue($isIceBox);
		self::assertTrue(session::get('isIceBox'));
		session::set('isIceBox',null);
		@unlink(rtrim(config::getWwwRoot(),'/').'/isIceBox');
		self::assertFileDoesNotExist(rtrim(config::getWwwRoot(),'/').'/isIceBox');
		$_SERVER['HTTP_HOST']='foo';
		$isIceBox=config::isIceBox();
		self::assertFalse($isIceBox);
		self::assertFalse(session::get('isIceBox'));
		session::set('isIceBox',null);
		$_SERVER['HTTP_HOST']='foo.icebox.icebear.fi';
		$isIceBox=config::isIceBox();
		self::assertTrue($isIceBox);
		self::assertTrue(session::get('isIceBox'));
	}

	public function testGetIceBoxName(){
		self::assertNull(session::get('iceBoxName'));
		config::getIceBoxName();
		self::assertNull(session::get('iceBoxName'));
		$_SERVER['HTTP_HOST']='foo.icebox.icebear.fi';
		self::assertEquals('foo', config::getIceBoxName());
		self::assertEquals('foo',session::get('iceBoxName'));
		$_SERVER['HTTP_HOST']=null;
		self::assertEquals('foo', config::getIceBoxName());
	}

}