<?php

include_once 'DeviceTest.php';

class UserManagementTest extends DeviceTest {

    const EMAIL='testbob@example.com';
    const FULL_NAME='Test Bob Smith';
    const USERNAME = 'TestBobSmith';

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateUserNoDetails(){
        $this->expectException('BadRequestException');
        UserManagement::createUser([]);
    }

    /**
     * @throws ServerException
     * @throws ForbiddenException
     * @throws NotFoundException
     */
    public function testCreateUserFullNameOnly(){
        $this->expectException('BadRequestException');
        UserManagement::createUser(
            array('fullName'=>self::FULL_NAME)
        );
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateUserEmailOnly(){
        $this->expectException('BadRequestException');
        UserManagement::createUser(
            array('email'=>self::EMAIL)
        );
    }

    /**
     * @throws ServerException
     * @throws NotFoundException
     * @throws ForbiddenException
     */
    public function testCreateUserUsernameOnly(){
        $this->expectException('BadRequestException');
        UserManagement::createUser(
            array('username'=>self::USERNAME)
        );
    }

    /**
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateUserEmailAndFullName(){
        $result=UserManagement::createUser(
            array('email'=>self::EMAIL, 'fullName'=>self::FULL_NAME)
        );
        self::assertEquals(self::EMAIL, $result['email']);
        self::assertEquals(self::FULL_NAME, $result['fullname']);
        self::assertEquals(0, $result['isactive']);
        self::assertMatchesRegularExpression('/^AUTO[0-9]{8}$/', $result['name']);
    }

    /**
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     */
    public function testCreateUserUsernameAndFullName(){
        $result=UserManagement::createUser(
            array('username'=>self::USERNAME, 'fullName'=>self::FULL_NAME)
        );
        self::assertEquals(self::USERNAME, $result['name']);
        self::assertEquals(self::FULL_NAME, $result['fullname']);
        self::assertEquals(0, $result['isactive']);
    }

    /**
     * @throws NotFoundException
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws ServerException
     */
    public function testCreateUserUsernameEmailAndFullName(){
        $result=UserManagement::createUser(
            array('email'=>self::EMAIL, 'username'=>self::USERNAME, 'fullName'=>self::FULL_NAME)
        );
        self::assertEquals(self::USERNAME, $result['name']);
        self::assertEquals(self::FULL_NAME, $result['fullname']);
        self::assertEquals(self::EMAIL, $result['email']);
        self::assertEquals(0, $result['isactive']);
    }

    /**
     * @throws ForbiddenException
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ServerException
     * @throws Exception
     */
    public function testCreateUserOverHttp(){
        $now='time'.time();
        $config = json_decode(@file_get_contents(__DIR__ . '/../../../../testconfig.json'), true);
        $apiRoot=$config['iceBearWwwRoot'];
        session::becomeAdmin();
        $apiKey=apikey::create(array('scope'=>'UserManagement'));
        $apiKeyId=$apiKey['created']['id'];
        $apiKey=$apiKey['created']['key'];
        database::commit();

        $payload=array('email'=>$now.self::EMAIL, 'username'=>$now.self::USERNAME, 'fullName'=>$now.self::FULL_NAME, 'csrfToken'=>session::get('csrfToken'));
        $result=DeviceTest::post($apiRoot.'/device/UserManagement/CreateUser',json_encode($payload), $apiKey);
        $result=json_decode($result,true);

        database::begin();
        apikey::delete($apiKeyId);
        database::commit();
        session::revokeAdmin();

        database::begin();

        self::assertIsArray($result);
        self::assertArrayNotHasKey("error",$result);
        self::assertEquals($now.self::USERNAME, $result['name']);
        self::assertEquals($now.self::FULL_NAME, $result['fullname']);
        self::assertEquals($now.self::EMAIL, $result['email']);
        self::assertEquals(0, $result['isactive']);
    }

    /**
     * @throws BadRequestException
     * @throws NotFoundException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testActivateUser(){
        session::becomeAdmin();
        $result=user::create(array(
            'email'=>self::EMAIL, 'name'=>self::USERNAME, 'fullname'=>self::FULL_NAME,
            'password'=>'dummy', 'isactive'=>0
        ))['created'];
        $userId=$result['id'];
        self::assertEquals(0, $result['isactive']);
        $result=UserManagement::activateUser(array('username'=>self::USERNAME));
        self::assertEquals(1, $result['isactive']);
        session::becomeAdmin(); //reverted by activateUser
        $user=user::getById($userId);
        self::assertEquals(1, $user['isactive']);
    }

    /**
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testDeactivateUser(){
        session::becomeAdmin();
        $result=user::create(array(
            'email'=>self::EMAIL, 'name'=>self::USERNAME, 'fullname'=>self::FULL_NAME,
            'password'=>'dummy', 'isactive'=>1
        ))['created'];
        $userId=$result['id'];
        self::assertEquals(1, $result['isactive']);
        $result=UserManagement::deactivateUser(array('username'=>self::USERNAME));
        self::assertEquals(0, $result['isactive']);
        session::becomeAdmin(); //reverted by activateUser
        $user=user::getById($userId);
        self::assertEquals(0, $user['isactive']);
    }

}
