<?php

use PHPUnit\Framework\TestCase;

class containerTest extends TestCase
{

	static int $now;
	private static mixed $type1;
	private static mixed $type2;

	/**
	 * @throws BadRequestException
	 * @throws ForbiddenException
	 * @throws ServerException
	 */
	protected function setUp(): void
    {
        database::connect();
		database::begin();
        try {
			$now=time();
			static::$now=$now;
            session::init(new DummySession());
            session::becomeAdmin();
			$user=user::create([
				'name'=>'user'.$now,
				'fullname'=>'user'.$now,
				'email'=>'user'.$now.'@bogus.bogus',
			])['created'];
            session::set('user', $user);
            session::set('userId', $user['id']);
			$category1=containercategory::create([
				'name'=>'1Category'.$now,
				'userscancreate'=>true
			])['created'];
			static::$type1=containertype::create([
				'name'=>'1Type'.$now,
				'containercategoryid'=>$category1['id'],
				'positions'=>'5'
			])['created'];
			$category2=containercategory::create([
				'name'=>'2Category'.$now,
				'userscancreate'=>false
			])['created'];
			static::$type2=containertype::create([
				'name'=>'2Type'.$now,
				'containercategoryid'=>$category2['id'],
				'positions'=>'5'
			])['created'];
        } catch (BadRequestException $e) {
            echo $e->getMessage();
			echo $e->getTraceAsString();
            $this->fail('BadRequestException on setup');
        } catch (NotFoundException $e) {
			echo $e->getTraceAsString();
            $this->fail('NotFoundException on setup');
        } catch (ServerException $e) {
			echo $e->getTraceAsString();
            $this->fail('ServerException on setup');
        }
        session::revokeAdmin();
    }

    protected function tearDown(): void {
        database::abort();
        session::destroy();
    }

	/**
	 * @throws BadRequestException
	 * @throws NotFoundException
	 * @throws ServerException
	 */
    public function testCanCreateAsAdmin(){
        session::becomeAdmin();
        self::assertTrue(container::canCreate(), 'container::canCreate() returned false as admin');
    }

    /**
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCanCreateAsTechnician(){
        session::becomeAdmin();
        $group=usergroup::getByName(usergroup::TECHNICIANS);
        groupmembership::create(array('userid'=>session::getUserId(), 'usergroupid'=>$group['id']));
        session::revokeAdmin();
        session::refreshProjectPermissions();
        self::assertTrue(container::canCreate(), 'containercategory::canCreate() returned false as technician');
    }

	/**
	 * @return void
	 * @throws BadRequestException
	 * @throws NotFoundException
	 * @throws ServerException
	 */
    public function testCanCreateAsNonAdmin(){
        self::assertTrue(container::canCreate(), 'container::canCreate() returned false as regular user');
    }

    /**
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateAsAdmin(){
        session::becomeAdmin();
		$created=container::create(array(
			'name'=>'Test1'.static::$now,
			'containertypeid'=>static::$type1['id']
		))['created'];
		self::assertNotEmpty($created);
		$created=container::create(array(
			'name'=>'Test2'.static::$now,
			'containertypeid'=>static::$type2['id']
		))['created'];
		self::assertNotEmpty($created);
        session::revokeAdmin();
    }

    /**
     * @throws NotFoundException
     * @throws BadRequestException
     * @throws ForbiddenException
     * @throws ServerException
     */
    public function testCreateAsTechnician(){
        session::becomeAdmin();
        $group=usergroup::getByName(usergroup::TECHNICIANS);
        groupmembership::create(array('userid'=>session::getUserId(), 'usergroupid'=>$group['id']));
        session::revokeAdmin();
        session::refreshProjectPermissions();
		$created=container::create(array(
			'name'=>'Test1'.static::$now,
			'containertypeid'=>static::$type1['id']
		))['created'];
		self::assertNotEmpty($created);
		$created=container::create(array(
			'name'=>'Test2'.static::$now,
			'containertypeid'=>static::$type2['id']
		))['created'];
		self::assertNotEmpty($created);
    }

	/**
	 * @throws NotFoundException
	 * @throws BadRequestException
	 * @throws ForbiddenException
	 * @throws ServerException
	 */
	public function testCreateAsNonAdmin(){
		$created=container::create(array(
			'name'=>'Test1'.static::$now,
			'containertypeid'=>static::$type1['id']
		))['created'];
		self::assertNotEmpty($created);
		session::revokeAdmin();
		self::expectException('ForbiddenException');
		container::create(array(
			'name'=>'Test2'.static::$now,
			'containertypeid'=>static::$type2['id']
		))['created'];
	}

}